/*********************************************************************

Author:     Tomas Franzon
Date:       2008-07-09
Program:    UserPort.SYS
Compile:    Use DDK BUILD facility

Purpose:    Give direct port I/O access to user mode programs.

This driver is influenced by an article written by Dale Roberts 8/30/95,
published in May 96 Dr Dobbs Journal, see www.ddj.com.
The driver gives user mode program access to selected ports by changing
the x86-processors IOPM (I/O Permission Map).

The driver tries to read the registry keys:
HKEY_LOCAL_MACHINE\Software\UserPort\IOPM
It will use the default values below if these doesn't exist.

History:
Version 1.0: 
*First release
Version 1.1:
* Added Test LPT1 button
* Added Test Spkr button
* Start button is greyed when the driver is started
* Stop button is greyed when the driver is stopped
* UserPort driver now works on Multiprocessor machines and Hyperthreading machines
* UserPort now works on all Windows XP machines
* \\.\UserPort access is now stopped when the \\.\UserPort file is closed
* Sleep(100) is no longer needed after the CreateFile("\\.\UserPort"...) call
* Address range is extended to 0x7a80 for all processes and to 0xffff for \\.\UserPort processes
* Added Java and VisualBasic example files
Version 2.0:
* Changed to PsSetCreateProcessNotifyRoutine method
* Removed the ugly and assembler modification of the TSS
  (not needed now with the process notification routine)
* Changed to only one IOPM registry map
* Address range is extended to 0xffff for all processes

*********************************************************************/
#include <ntddk.h>

/*
 *  The name of our device driver.
 */
#define DEVICE_NAME_STRING L"UserPort"

/*
  OriginalMapCopy is used to restore the IOPM when the driver exists.
  CurrentMap points to NULL or to the place where the processors IOPM is located.
  Accessmap is the IOPM that is used by the driver.
  Every port address has one cooresponding bit in the IOPM. The driver supports
  addresses from 0x000 to 0x3FF and the IOPM size is then 0x3ff / 8 = 0x80.
 */

UCHAR OriginalIOPMCopy[0x2004];
UCHAR NewIOPMBuffer[0x2004+sizeof(KEY_VALUE_PARTIAL_INFORMATION)];
const UCHAR DefaultMap[0x80]=
/*0x000*/ {0xFF,0xFF,
/*0x010*/  0xFF,0xFF,
/*0x020*/  0xFF,0xFF,
/*0x030*/  0xFF,0xFF,
/*0x040*/  0xFF,0xFF,
/*0x050*/  0xFF,0xFF,
/*0x060*/  0xFF,0xFF,
/*0x070*/  0xFF,0xFF,
/*0x080*/  0xFF,0xFF,
/*0x090*/  0xFF,0xFF,
/*0x0A0*/  0xFF,0xFF,
/*0x0B0*/  0xFF,0xFF,
/*0x0C0*/  0xFF,0xFF,
/*0x0D0*/  0xFF,0xFF,
/*0x0E0*/  0xFF,0xFF,
/*0x0F0*/  0xFF,0xFF,
/*0x100*/  0xFF,0xFF,
/*0x110*/  0xFF,0xFF,
/*0x120*/  0xFF,0xFF,
/*0x130*/  0xFF,0xFF,
/*0x140*/  0xFF,0xFF,
/*0x150*/  0xFF,0xFF,
/*0x160*/  0xFF,0xFF,
/*0x170*/  0xFF,0xFF,
/*0x180*/  0xFF,0xFF,
/*0x190*/  0xFF,0xFF,
/*0x1A0*/  0xFF,0xFF,
/*0x1B0*/  0xFF,0xFF,
/*0x1C0*/  0xFF,0xFF,
/*0x1D0*/  0xFF,0xFF,
/*0x1E0*/  0xFF,0xFF,
/*0x1F0*/  0xFF,0xFF,
/*0x200*/  0x00,0x00,
/*0x210*/  0x00,0x00,
/*0x220*/  0x00,0x00,
/*0x230*/  0x00,0x00,
/*0x240*/  0x00,0x00,
/*0x250*/  0x00,0x00,
/*0x260*/  0x00,0x00,
/*0x270*/  0x00,0x00,
/*0x280*/  0x00,0x00,
/*0x290*/  0x00,0x00,
/*0x2A0*/  0x00,0x00,
/*0x2B0*/  0x00,0x00,
/*0x2C0*/  0x00,0x00,
/*0x2D0*/  0x00,0x00,
/*0x2E0*/  0x00,0x00,
/*0x2F0*/  0x00,0x00,
/*0x300*/  0x00,0x00,
/*0x310*/  0x00,0x00,
/*0x320*/  0x00,0x00,
/*0x330*/  0x00,0x00,
/*0x340*/  0x00,0x00,
/*0x350*/  0x00,0x00,
/*0x360*/  0x00,0x00,
/*0x370*/  0x00,0x00,
/*0x380*/  0xFF,0xFF,
/*0x390*/  0xFF,0xFF,
/*0x3A0*/  0xFF,0xFF,
/*0x3B0*/  0xFF,0x0F,
/*0x3C0*/  0xFF,0xFF,
/*0x3D0*/  0xFF,0xFF,
/*0x3E0*/  0xFF,0x00,
/*0x3F0*/  0x00,0x00};

/*
 Ke386IoSetAccessProcess() adjusts the IOPM offset pointer to the IOPM at 0x88
 Ke386IoSetAccessProcess() is located in NTOSKRNL.EXE but is not included in any
 header file or documented anywhere...
*/

void Ke386IoSetAccessProcess(PEPROCESS, int);
void Ke386SetIoAccessMap(int, UCHAR *);
void Ke386QueryIoAccessMap(int, UCHAR *);
NTSTATUS PsLookupProcessByProcessId(IN HANDLE ProcessId, OUT PEPROCESS *Process);
NTSYSAPI NTSTATUS NTAPI ZwYieldExecution(VOID);


/*********************************************************************
  This routine is called every time a new process is created. This is
  where UserPort gets activated by swithing the IOPM
*********************************************************************/
VOID CreateProcessNotifyRoutine(
    IN HANDLE  ParentId,
    IN HANDLE  ProcessId,
    IN BOOLEAN  Create
    )
{
  PEPROCESS Process=NULL;
  
  // Give the new process IO access
  PsLookupProcessByProcessId(ProcessId, &Process);
  if (Process != NULL) {
    Ke386IoSetAccessProcess(Process, 1);
    ZwYieldExecution(); // Force a process switch
  }
}

/*********************************************************************
  Service handler for a CreateFile() user mode call.

  This routine is entered in the driver object function call table by
the DriverEntry() routine.  When the user mode application calls
CreateFile(), this routine gets called while still in the context of
the user mode application, but with the CPL (the processor's Current
Privelege Level) set to 0.  This allows us to do kernel mode
operations.  UserPort() is called to give the calling process I/O
access.  All the user mode application needs do to obtain I/O access
is open this device with CreateFile().  No other operations are
required.
Note: This CreateFile handling is kept from older versions of UserPort
It should not be needed in UserPort2.0 but does no harm
*********************************************************************/
NTSTATUS CreateFileDispatch(
    IN  PDEVICE_OBJECT  DeviceObject,
    IN  PIRP            Irp
    )
{
  // Give the current process IO access
  Ke386IoSetAccessProcess(PsGetCurrentProcess(), 1);
  ZwYieldExecution(); // Force a process switch

  Irp->IoStatus.Information = 0;
  Irp->IoStatus.Status = STATUS_SUCCESS;
  IoCompleteRequest(Irp, IO_NO_INCREMENT);
  return STATUS_SUCCESS;
}

NTSTATUS CloseFileDispatch(
    IN  PDEVICE_OBJECT  DeviceObject,
    IN  PIRP            Irp
    )
{
  // Deny the current process IO access
  Ke386IoSetAccessProcess(PsGetCurrentProcess(), 0);
  ZwYieldExecution(); // Force a process switch

  Irp->IoStatus.Information = 0;
  Irp->IoStatus.Status = STATUS_SUCCESS;
  IoCompleteRequest(Irp, IO_NO_INCREMENT);
  return STATUS_SUCCESS;
}

// remove the link \\.\UserPort and restore the IOPM 
void UserPortUnload(IN  PDRIVER_OBJECT  DriverObject)
{
  WCHAR DOSNameBuffer[] = L"\\DosDevices\\" DEVICE_NAME_STRING;
  UNICODE_STRING uniDOSString;

  // Restore to the original map
  Ke386SetIoAccessMap(1, OriginalIOPMCopy);

  PsSetCreateProcessNotifyRoutine(CreateProcessNotifyRoutine,TRUE);

  RtlInitUnicodeString(&uniDOSString, DOSNameBuffer);
  IoDeleteSymbolicLink (&uniDOSString);
  IoDeleteDevice(DriverObject->DeviceObject);
}

// This routine is the entrypoint of the driver.
// This routine reads the IOPM from registry and start the driver
NTSTATUS DriverEntry(
    IN PDRIVER_OBJECT DriverObject,
    IN PUNICODE_STRING RegistryPath
    )
{
  PDEVICE_OBJECT deviceObject;
  NTSTATUS status;
  WCHAR NameBuffer[] = L"\\Device\\" DEVICE_NAME_STRING;
  WCHAR DOSNameBuffer[] = L"\\DosDevices\\" DEVICE_NAME_STRING;
  UNICODE_STRING uniNameString, uniDOSString;
  UCHAR *IOPM;
  unsigned i;

  ULONG nInformation1;
  PKEY_VALUE_PARTIAL_INFORMATION Information1;
  ULONG ResultLength;
  HANDLE KeyHandle;
  OBJECT_ATTRIBUTES ObjectAttributes;
  UNICODE_STRING IOPMString,RegPathString;

  RtlInitUnicodeString(&IOPMString, L"IOPM");
  RtlInitUnicodeString(&RegPathString,L"\\Registry\\Machine\\System\\CurrentControlSet\\Services\\UserPort");

  InitializeObjectAttributes(&ObjectAttributes,&RegPathString,OBJ_CASE_INSENSITIVE,NULL,NULL);

  nInformation1 = sizeof(NewIOPMBuffer);
  Information1 = (PKEY_VALUE_PARTIAL_INFORMATION)NewIOPMBuffer;

  RtlFillMemory(Information1, nInformation1, 0xFF);
  Information1->Type = REG_BINARY;
  Information1->DataLength = 0x2000;  
  IOPM = Information1->Data;

  if (STATUS_SUCCESS == ZwOpenKey(&KeyHandle, KEY_QUERY_VALUE,&ObjectAttributes)) {
    if (STATUS_SUCCESS != ZwQueryValueKey(KeyHandle,&IOPMString,KeyValuePartialInformation,Information1,nInformation1,&ResultLength)) {
      RtlCopyMemory(IOPM, DefaultMap, sizeof(DefaultMap));
    }

    ZwClose(KeyHandle);
  }
  else {
    RtlCopyMemory(IOPM, DefaultMap, sizeof(DefaultMap));
  }

  // Update the IOPM map
  Ke386QueryIoAccessMap(1, OriginalIOPMCopy);
  for (i=0;i<sizeof(OriginalIOPMCopy);i++) {
    // There might be an other driver using the IOPM so lets merge them together
    IOPM[i] = OriginalIOPMCopy[i] & IOPM[i];    
  }
  Ke386SetIoAccessMap(1, IOPM);

  //
  // Set up device driver name and device object.
  // Make the driver accessable though the file \\.\UserPort
	
  RtlInitUnicodeString(&uniNameString, NameBuffer);
  RtlInitUnicodeString(&uniDOSString, DOSNameBuffer);

  status = IoCreateDevice(DriverObject,0,&uniNameString,FILE_DEVICE_UNKNOWN,0, FALSE, &deviceObject);
  if(!NT_SUCCESS(status))
    return status;

  status = IoCreateSymbolicLink (&uniDOSString, &uniNameString);

  if (!NT_SUCCESS(status))
    return status;

  //
  // Initialize the Driver Object with driver's entry points.
  // All we require are the Create and Unload operations.
  //
  DriverObject->MajorFunction[IRP_MJ_CREATE] = CreateFileDispatch;
  DriverObject->MajorFunction[IRP_MJ_CLOSE] = CloseFileDispatch;

  DriverObject->DriverUnload = UserPortUnload;
  
  PsSetCreateProcessNotifyRoutine(CreateProcessNotifyRoutine,FALSE);


  return STATUS_SUCCESS;
}

